;;; ------------------------------------------------------------
;;;  TEXTEXPORT.LSP  (Version mit MLEADER)
;;;  Exportiert TEXT, MTEXT, ATTRIB und MLEADER nach CSV
;;;  Struktur: ID;SpaceType;LayoutName;EntityType;BlockName;AttrTag;Text
;;; ------------------------------------------------------------

(vl-load-com)

(defun td:csv-safe (s /)
  ;; Wandelt problematische Zeichen in Leerzeichen um,
  ;; damit die CSV-Struktur nicht kaputtgeht.
  (if (null s) (setq s ""))
  (vl-string-translate "\n\r;\"" "    " s)
)

(defun c:TEXTEXPORT (/ doc layouts lay layName spaceType btr obj oname
                       blkName attTag txt id f prefix name filepath row attList)

  (vl-load-com)

  (setq doc      (vla-get-ActiveDocument (vlax-get-Acad-Object))
        layouts  (vla-get-Layouts doc)
        prefix   (getvar "dwgprefix")
        name     (vl-filename-base (getvar "dwgname"))
        filepath (strcat prefix name "_TextExport.csv")
  )

  (setq f (open filepath "w"))

  (if (null f)
    (prompt "\n*** Fehler: Exportdatei konnte nicht erstellt werden. ***")
    (progn
      ;; Kopfzeile schreiben
      (write-line
        "ID;SpaceType;LayoutName;EntityType;BlockName;AttrTag;Text"
        f
      )

      (setq id 0)

      ;; Alle Layouts durchlaufen (inkl. Model-Space)
      (vlax-for lay layouts
        (setq layName   (vla-get-Name lay)
              spaceType (if (= (strcase layName) "MODEL")
                          "Model"
                          "Layout"
                        )
              btr       (vla-get-Block lay)
        )

        ;; Alle Objekte in diesem Layout/Space durchlaufen
        (vlax-for obj btr
          (setq oname (vla-get-ObjectName obj))

          (cond
            ;; TEXT
            ((= oname "AcDbText")
             (setq id  (1+ id)
                   txt (td:csv-safe (vla-get-TextString obj))
                   row (strcat
                         (itoa id) ";"
                         spaceType ";"
                         (td:csv-safe layName) ";"
                         "TEXT;"
                         ";"
                         ";"
                         txt
                       )
             )
             (write-line row f)
            )

            ;; MTEXT (als eine Zeile)
            ((= oname "AcDbMText")
             (setq id  (1+ id)
                   txt (td:csv-safe (vla-get-TextString obj))
                   row (strcat
                         (itoa id) ";"
                         spaceType ";"
                         (td:csv-safe layName) ";"
                         "MTEXT;"
                         ";"
                         ";"
                         txt
                       )
             )
             (write-line row f)
            )

            ;; MLEADER (MF-Linie) – Textinhalt exportieren
            ((= oname "AcDbMLeader")
             (setq id  (1+ id)
                   txt (td:csv-safe (vla-get-TextString obj))
                   row (strcat
                         (itoa id) ";"
                         spaceType ";"
                         (td:csv-safe layName) ";"
                         "MLEADER;"
                         ";"
                         ";"
                         txt
                       )
             )
             (write-line row f)
            )

            ;; Blockreferenzen: Attribute exportieren (fürs Excel ok),
            ;; werden beim Import später bewusst ignoriert.
            ((= oname "AcDbBlockReference")
             (if (= (vla-get-HasAttributes obj) :vlax-true)
               (progn
                 (setq blkName (td:csv-safe (vla-get-Name obj)))
                 (setq attList
                       (vlax-safearray->list
                         (vlax-variant-value
                           (vla-GetAttributes obj)
                         )
                       )
                 )
                 (foreach att attList
                   (setq id     (1+ id)
                         attTag (td:csv-safe (vla-get-TagString  att))
                         txt    (td:csv-safe (vla-get-TextString att))
                         row    (strcat
                                  (itoa id) ";"
                                  spaceType ";"
                                  (td:csv-safe layName) ";"
                                  "ATTRIB;"
                                  blkName ";"
                                  attTag  ";"
                                  txt
                                )
                   )
                   (write-line row f)
                 )
               )
             )
            )
          ) ; cond
        ) ; vlax-for obj
      ) ; vlax-for lay

      (close f)

      (prompt
        (strcat
          "\nTextexport abgeschlossen: "
          filepath
        )
      )
    )
  )

  (princ)
)
